<?php
/**
 * Theme functions and definitions.
 *
 * For additional information on potential customization options,
 * read the developers' documentation:
 *
 * https://developers.elementor.com/docs/hello-elementor-theme/
 *
 * @package HelloElementorChild
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

define( 'HELLO_ELEMENTOR_CHILD_VERSION', '2.0.0' );

/**
 * Load child theme scripts & styles.
 *
 * @return void
 */
function hello_elementor_child_scripts_styles() {

	wp_enqueue_style(
		'hello-elementor-child-style',
		get_stylesheet_directory_uri() . '/style.css',
		[
			'hello-elementor-theme-style',
		],
		HELLO_ELEMENTOR_CHILD_VERSION
	);

}
add_action( 'wp_enqueue_scripts', 'hello_elementor_child_scripts_styles', 20 );

function load_font_awesome_cdn() {
    wp_enqueue_style( 'font-awesome-cdn', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css' );
}
add_action( 'wp_enqueue_scripts', 'load_font_awesome_cdn' );


/* Disabled default guttenburg editor */
add_filter('use_block_editor_for_post', '__return_false');

function scroll_logo_swap_script() {
    if ( ! is_admin() ) {
        wp_register_script('scroll-logo-swap', '', [], '', true);
        wp_enqueue_script('scroll-logo-swap');

        $script = <<<JS
document.addEventListener('DOMContentLoaded', function () {
    const header = document.getElementById('logo-main');
    if (!header) return;
    const logoImg = header.querySelector('img');

    const original = {
        src: 'https://datasense.joomxone.com/wp-content/uploads/2025/07/datasense-logo-white.png',
        srcset: 'https://datasense.joomxone.com/wp-content/uploads/2025/07/datasense-logo-white.png 320w, https://datasense.joomxone.com/wp-content/uploads/2025/07/datasense-logo-white-300x72.png 300w',
        sizes: '(max-width: 320px) 100vw, 320px'
    };

    const scrolled = {
        src: 'https://datasense.joomxone.com/wp-content/uploads/2025/07/datasense-logo-demo.png',
        srcset: '', // Leave empty or use correct responsive versions if you have them
        sizes: ''
    };

    let current = 'original';

    window.addEventListener('scroll', function () {
        const scrollY = window.scrollY;

        if (scrollY > 50 && current !== 'scrolled') {
            logoImg.src = scrolled.src;
            logoImg.srcset = scrolled.srcset;
            logoImg.sizes = scrolled.sizes;
            current = 'scrolled';
        } else if (scrollY <= 50 && current !== 'original') {
            logoImg.src = original.src;
            logoImg.srcset = original.srcset;
            logoImg.sizes = original.sizes;
            current = 'original';
        }
    });
});
JS;


        wp_add_inline_script('scroll-logo-swap', $script);
    }
}
add_action('wp_enqueue_scripts', 'scroll_logo_swap_script');


add_filter('acf/shortcode/allow_in_block_themes_outside_content', '__return_true');

add_action( 'acf/init', 'set_acf_settings' );
function set_acf_settings() {
acf_update_setting( 'enable_shortcode', true );
}


// Shortcode to expose post ID for Teams
add_shortcode('post_id', function() {
    return get_the_ID();
});


add_action('wp_ajax_load_post_details', 'load_post_details');
add_action('wp_ajax_nopriv_load_post_details', 'load_post_details');

function load_post_details() {
    if (!isset($_POST['post_id']) || !is_numeric($_POST['post_id'])) {
        wp_send_json_error('Invalid or missing post_id', 400);
    }

    $post_id = intval($_POST['post_id']);
    $post = get_post($post_id);

    if (!$post) {
        wp_send_json_error('Post not found', 404);
    }

    $photo = get_the_post_thumbnail_url($post_id, 'medium');
    $name = get_the_title($post_id);
    $designation = get_field('team_designation', $post_id);
    $bio = get_field('team_brief', $post_id);
    $linkedin = get_field('team_linkedin', $post_id);
    $email = get_field('team_email', $post_id);

    echo '<div class="popup-profile">';
    echo    '<img src="' . esc_url($photo) . '" alt="' . esc_attr($name) . '" class="profile-image">';
    echo    '<h2>' . esc_html($name) . '</h2>';
    echo    '<p class="designation">' . esc_html($designation) . '</p>';
	echo '<div class="social-links">';
	if ($linkedin) {
		echo '<a href="' . esc_url($linkedin) . '" target="_blank"><i class="fab fa-linkedin-in"></i></a>';
	}
	if ($email) {
		echo '<a href="mailto:' . sanitize_email($email) . '"><i class="fas fa-envelope"></i></a>';
	}
	echo '</div>';

    echo    '<div class="bio">' . wp_kses_post(wpautop($bio)) . '</div>';
    echo '</div>';

    wp_die();
}

/*function custom_post_id_shortcode() {
    return get_the_ID();
}
add_shortcode( 'post_id', 'custom_post_id_shortcode' );

add_action('wp_ajax_load_post_details', 'load_post_details');
add_action('wp_ajax_nopriv_load_post_details', 'load_post_details');

function load_post_details() {
    $post_id = intval($_POST['post_id']);
    $post = get_post($post_id);

    if ($post) {
        echo '<h2>' . esc_html($post->post_title) . '</h2>';
        echo '<div>' . apply_filters('the_content', $post->post_content) . '</div>';
        echo get_the_post_thumbnail($post_id, 'large');
		echo get_field('team_designation', $post_id);
		echo get_field('team_brief', $post_id);
    } else {
        echo 'Post not found.';
    }
    wp_die();
}*/

//Sorting dropdown shortcode
add_shortcode('simple_sort', function () {
    global $wp;
    $url = home_url(add_query_arg([], $wp->request));

    $sort = isset($_GET['sort']) ? sanitize_text_field($_GET['sort']) : 'date_DESC';

    ob_start(); ?>
    <form class="simple-sort-form" action="<?php echo esc_url($url); ?>" method="get">
        <label for="sort" class="sr-only">Sort posts</label>
        <select id="sort" name="sort" onchange="this.form.submit()">
            <option value="date_DESC"  <?php selected($sort, 'date_DESC'); ?>>Sort By Date (Latest)</option>
            <option value="date_ASC"   <?php selected($sort, 'date_ASC');  ?>>Sort By Date (Oldest)</option>
            <option value="title_ASC"  <?php selected($sort, 'title_ASC'); ?>>Sort By Keyword (A–Z)</option>
        </select>
        <?php
        // Preserve the filter_tag in form submission
        if (isset($_GET['filter_tag']) && $_GET['filter_tag'] !== '') {
            echo '<input type="hidden" name="filter_tag" value="' . esc_attr($_GET['filter_tag']) . '">';
        }
        ?>
    </form>
    <?php
    return ob_get_clean();
});

// Tag filter shortcode
add_shortcode('tag_filter', function() {
    $tags = get_tags();
    if (!$tags) return '';

    $current_tag = isset($_GET['filter_tag']) ? sanitize_text_field($_GET['filter_tag']) : '';

    ob_start();
    echo '<div class="tag-filter-buttons">';
    echo '<button class="tag-filter-btn' . ($current_tag === '' ? ' active' : '') . '" data-tag="">All</button>';
    foreach ($tags as $tag) {
        $active = ($current_tag === $tag->slug) ? ' active' : '';
        echo '<button class="tag-filter-btn' . $active . '" data-tag="' . esc_attr($tag->slug) . '">' . esc_html($tag->name) . '</button>';
    }
    echo '</div>';
    return ob_get_clean();
});

// Apply sorting + tag filter to main query
add_action('pre_get_posts', function ($query) {
    if (!is_admin() && $query->is_main_query()) {

        // Sorting
        if (isset($_GET['sort'])) {
            $sort = sanitize_text_field($_GET['sort']);
            switch ($sort) {
                case 'date_DESC':
                    $query->set('orderby', 'date');
                    $query->set('order', 'DESC');
                    break;
                case 'date_ASC':
                    $query->set('orderby', 'date');
                    $query->set('order', 'ASC');
                    break;
                case 'title_ASC':
                    $query->set('orderby', 'title');
                    $query->set('order', 'ASC');
                    break;
            }
        }

        // Filtering by tag
        if (isset($_GET['filter_tag']) && $_GET['filter_tag'] !== '') {
            $tag = sanitize_text_field($_GET['filter_tag']);
            $query->set('tag', $tag);
        }
    }
});


add_action('elementor/query/team_sort', function($query) {
    $query->set('meta_key', 'team_order'); 
    $query->set('orderby', 'meta_value_num');
    $query->set('order', 'ASC');
});

// for publications post type url
function custom_publication_post_type_args( $args, $post_type ) {
    if ( 'publications' === $post_type ) {
        $args['rewrite'] = [
            'slug' => 'publications/%publication-type%',
            'with_front' => false,
        ];
    }
    return $args;
}
add_filter( 'register_post_type_args', 'custom_publication_post_type_args', 10, 2 );

function publication_post_type_link( $post_link, $post ) {
    if ( 'publications' === get_post_type( $post ) ) {
        $terms = get_the_terms( $post->ID, 'publication-type' );
        if ( $terms && ! is_wp_error( $terms ) ) {
            $slug = $terms[0]->slug;
            return str_replace( '%publication-type%', $slug, $post_link );
        } else {
            return str_replace( '%publication-type%', 'uncategorized', $post_link );
        }
    }
    return $post_link;
}
add_filter( 'post_type_link', 'publication_post_type_link', 10, 2 );


add_action( 'elementor/query/publication_cat', function( $query ) {
    if ( is_tax( 'publication-type' ) ) {
        $term = get_queried_object();
        $query->set( 'tax_query', [
            [
                'taxonomy' => 'publication-type',
                'field'    => 'slug',
                'terms'    => $term->slug,
            ]
        ]);
    }
} );